<?php

namespace App\Livewire;

use App\Models\MaintenanceRecord;
use App\Models\Vehicle;
use Illuminate\Support\Facades\DB;
use Livewire\Component;
use Livewire\Attributes\Computed;

class MaintenanceStatusWidget extends Component
{
    #[Computed]
    public function upcomingServicesByDate()
    {
        return MaintenanceRecord::with('vehicle')
            ->whereNotNull('next_service_date')
            ->whereBetween('next_service_date', [now(), now()->addDays(30)])
            ->orderBy('next_service_date', 'asc')
            ->get();
    }

    #[Computed]
    public function overdueServicesByDate()
    {
        return MaintenanceRecord::with('vehicle')
            ->whereNotNull('next_service_date')
            ->where('next_service_date', '<', now())
            ->orderBy('next_service_date', 'desc')
            ->get();
    }

    #[Computed]
    public function dueServicesByOdometer()
    {
        // This query finds the latest maintenance record for each vehicle
        // that has a 'next_service_odometer' set, and then joins it
        // back to the vehicles table to compare with the 'current_odometer'.

        // Subquery to get the ID of the latest maintenance record for each vehicle
        $latestMaintenanceSubquery = MaintenanceRecord::select('vehicle_id', DB::raw('MAX(id) as id'))
            ->whereNotNull('next_service_odometer')
            ->groupBy('vehicle_id');

        return MaintenanceRecord::with('vehicle')
            ->joinSub($latestMaintenanceSubquery, 'latest_maintenance', function ($join) {
                $join->on('maintenance_records.id', '=', 'latest_maintenance.id');
            })
            ->join('vehicles', 'maintenance_records.vehicle_id', '=', 'vehicles.id')
            ->whereNotNull('maintenance_records.next_service_odometer')
            ->where('vehicles.current_odometer', '>=', 'maintenance_records.next_service_odometer')
            ->select('maintenance_records.*')
            ->get();
    }

    public function render()
    {
        return view('livewire.maintenance-status-widget');
    }
}